//
//  RMCurlURLConnection.h
//  FTP
//
//  Created by Keith Duncan on 22/01/2011.
//  Copyright 2011 Realmac Software. All rights reserved.
//

#import <Cocoa/Cocoa.h>

@class RMFTPURLCredential;

extern NSString *const RMCurlURLConnectionErrorDomain;

enum {
	RMCurlURLConnectionErrorCodeUnknown = 0,
	
	/* Authentication [-100, -199] */
	RMCurlURLConnectionErrorCodeInvalidCredentials = -100,
	
	/* Configuration [-200, -299] */
	RMCurlURLConnectionErrorCodeInvalidSettings = -200,
	RMCurlURLConnectionErrorCodeInvalidHostname = -201,
};

typedef NS_ENUM(NSInteger, RMCurlURLConnectionMode) {
	RMCurlURLConnectionModeExtendedPassive = 0,
	RMCurlURLConnectionModePassive = 1,
	RMCurlURLConnectionModeActive = 2,
};

typedef NSInteger RMCurlURLConnectionErrorCode;

extern NSString *const RMCurlURLConnectionDidProgressNotificationName;

	extern NSString *const RMCurlURLConnectionDidProgressNotificationBytesUploadedKey; // double
	extern NSString *const RMCurlURLConnectionDidProgressNotificationTotalBytesUploadedKey; // double
	extern NSString *const RMCurlURLConnectionDidProgressNotificationTotalBytesExpectedToUploadKey; // double

	extern NSString *const RMCurlURLConnectionDidProgressNotificationBytesDownloadedKey; // double
	extern NSString *const RMCurlURLConnectionDidProgressNotificationTotalBytesDownloadedKey; // double
	extern NSString *const RMCurlURLConnectionDidProgressNotificationTotalBytesExpectedToDownloadKey; // double

/*!
	\brief
	libcurl wrapper, providing FTP, FTPS and SFTP file management
	
	\details
	These methods can be invoked in either synchronous or asynchronous (with respect to the caller) mode.
	Invoking the methods on this connection with a nil `responseQueue` will cause them to run synchronously in the calling environment.
	Invoking the methods with a non nil `responseQueue` will cause them to run synchronously in the background and report completion to the caller asynchronously.
 */
@interface RMCurlURLConnection : NSOperation

/*!
	\brief
	Designated initialiser
 */
- (id)initWithURL:(NSURL *)URL;

@property (readonly, copy, nonatomic) NSURL *URL;

@property (retain, nonatomic) RMFTPURLCredential *credential;
@property (nonatomic, assign) BOOL ignoreCertificateErrors;
@property (nonatomic, assign) RMCurlURLConnectionMode connectionMode;
@property (nonatomic, assign) NSInteger useSSLOption;

@property (readonly, copy, atomic) id (^completionProvider)(NSError **);

@end

/*!
	\brief
	Simply attempt to sign in to the remote server to check credentials
 */
@interface RMCurlValidateCredentialOperation : RMCurlURLConnection

@end

/*!
	\brief
	Create remote directories
 */
@interface RMCurlCreateDirectoryOperation : RMCurlURLConnection

- (id)initWithURL:(NSURL *)URL directoryName:(NSString *)diretoryName;

@end

/*!
	\brief
	Create remote files
 */
@interface RMCurlCreateFileOperation : RMCurlURLConnection

- (id)initWithURL:(NSURL *)URL fileName:(NSString *)fileName fileLocation:(NSURL *)fileLocation;

@property (readonly, copy, atomic) NSURL * (^completionProvider)(NSError **);

@end

@interface RMCurlCreateFileFromSymlinkOperation : RMCurlCreateFileOperation

@end

/*!
	\brief
	Create multiple remote files in the same directory
	
	\details
	Use of this operation is a performance enhancement by eliminating redundant CWD commands to the server
 */
@interface RMCurlCreateFilesOperation : NSOperation

/*!
	\brief
	All operations added to this operation must have the same base URL.
	
	\details
	This method is not thread safe.
	All operations must be added to this operation before it's added to a queue.
	Results are undefined if operations added to this operation after adding it to a queue.
 */
- (void)addExecutionOperation:(RMCurlURLConnection *)operation;

@property (readonly, copy, atomic) id (^completionProvider)(NSError **);
@property (nonatomic, readonly) NSURL *URL;
@property (readonly, nonatomic) NSArray *executionOperations;

@end
